#!/bin/ksh 

#
# snmp.sh
# this script will gather the SNMP information about clusters defined in clhosts
# author: AIX Health Check
# Version 1.6
# Date: March 24, 2015
#

# Variable declarations

# Where will by script run from:
export BASE=/tmp/cl

# Where can i find the SNMP MIB for HACM?
export HASNMPHAMIB="$BASE/hacmp.defs"

#
# Function retrievehainfo will gather information for a single cluster
#

function retrievehainfo {

unset CLUSTER
unset INDMZ
CLUSTER=$1
INDMZ=$2
SNMPCOMMUNITY=`grep ^cluster:$CLUSTER: clhosts | cut -f6 -d:`

# Necessary for DMZ clusters. Works only with HACMP ES.
export DEFS=/usr/es/sbin/cluster/hacmp.defs

# Which interfaces are in this cluster
interfaces=`grep "^interface:$CLUSTER:" $BASE/clhosts | awk -F":" '{print $3}'`

rm -f $BASE/snmp/interfacefile
touch $BASE/snmp/interfacefile

# Check via a ping if a interface responds

if [ -z "$INDMZ" ] ; then

   # Cluster not in a DMZ, use a regular ping
   echo "$interfaces" | while read interface ; do
      unset pingresult
      pingresult=`ping -c 1 -w 1 $interface | grep -i " 0% packet loss"`
      if [ ! -z "$pingresult" ] ; then
         echo $interface >> $BASE/snmp/interfacefile
      fi
   done

else

   # Cluster within DMZ, use ssh_ping
   echo "$interfaces" | while read interface ; do
      ssh_ping -c 2 $interface | grep INFO | awk '{print $2}' >> $BASE/snmp/interfacefile
   done

fi

# If we've found any pingable addresses, we'll continue.
if [ -s $BASE/snmp/interfacefile ]
then
   unset teller
   # determine the amount of interfaces found
   let teller=`wc -l $BASE/snmp/interfacefile | awk '{print $1}'`

   # If there is only 1 found, we'll use that one.
   if [ $teller -eq 1 ]
   then
      export HASNMPHOST=`cat $BASE/snmp/interfacefile`
      if [ ! -z "$INDMZ" ]
      then
         ssh $HASNMPHOST "snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $DEFS clusterid clustername clusterstate clustersubstate clstrmgrversion clstrmgrVersion" > $BASE/snmp/$CLUSTER.clusterinfo 2>/dev/null
      else
         snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $HASNMPHAMIB clusterid clustername clusterstate clustersubstate clstrmgrversion clstrmgrVersion > $BASE/snmp/$CLUSTER.clusterinfo 2>/dev/null
      fi

      # If the file is empty, then quit. Our only interface won't work.
      # Bad luck.
      #if [ ! -s $BASE/snmp/$CLUSTER.clusterinfo ]
      #then
      #   exit
      #fi

   else

      # We have multiple interfaces, pick one using a loop.
      # Just in case a pingable address doesn't respond.
      cat $BASE/snmp/interfacefile | while read HASNMPHOST
      do
         if [ ! -z "$INDMZ" ]
         then
            result=`ssh $HASNMPHOST "snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $DEFS clusterid 2>/dev/null`
         else
            result=`snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $HASNMPHAMIB clusterid 2>/dev/null`
         fi
         if [ ! -z "$result" ]
         then
            if [ ! -z "$INDMZ" ]
            then
               ssh $HASNMPHOST "snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $DEFS clusterid clustername clusterstate clustersubstate clstrmgrversion clstrmgrVersion" > $BASE/snmp/$CLUSTER.clusterinfo 2>/dev/null
            else
               snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $HASNMPHAMIB clusterid clustername clusterstate clustersubstate clstrmgrversion clstrmgrVersion > $BASE/snmp/$CLUSTER.clusterinfo 2>/dev/null
            fi

            # Test the contents
            unset result
            result=`wc -l $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $1}'`
            if [ $result -gt 1 ]
            then
               # Okay, get out of this loop
               break
            fi
         else
            # If all interfaces failed, create an empty file
            touch $BASE/snmp/$CLUSTER.clusterinfo
         fi
      done

   fi

   # Create $cluster.out file

   # info: cluster id, cluster name and cluster status
   # clusterstate: 2 = up, 4 = down, 8 = unknown
   # clustersubstate: 16 = unstable, 64 = error, 32 = stable, 8 = unknown, 128 = reconfig

   echo Cluster ID: `grep -i clusterid $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $3}'`
   echo Cluster Name: `grep -i clustername $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $3}' | sed "s/\"//g"`
   unset clusterstate
   clusterstate=`grep -i clusterstate $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $3}'`
   case $clusterstate in 
      2) echo Cluster State: UP ;;
      4) echo Cluster State: DOWN ;;
      8) echo Cluster State: UNKNOWN ;;
   esac
   unset clustersubstate
   clustersubstate=`grep -i clustersubstate $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $3}'`
   case $clustersubstate in 
      16) echo Cluster SubState: UNSTABLE ;;
      64) echo Cluster SubState: ERROR ;;
      32) echo Cluster SubState: STABLE ;;
      8) echo Cluster SubState: UNKNOWN ;;
      128) echo Cluster SubState: RECONFIG ;;
   esac
   unset version
   version=`grep -i clstrmgrversion $BASE/snmp/$CLUSTER.clusterinfo | awk '{print $3}' | sort | head -1 | sed "s/\"//g"`
   if [ ! -z "$version" ] ; then
      echo Version: $version
   else
      # no version found in hacmp; this is a bug in ha 51, 53 and 54.
      # try to find out the hacmp version via another method
      version=`snmpinfo -m dump -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o /etc/mib.defs hrSWInstalled | grep cluster | grep server | grep rte | grep hrSWInstalledName | head -1 | awk '{print $4}' | cut -c1-3`
      if [ ! -z "$version" ] ; then
        echo Version: $version
      else
        echo Version: 
      fi
   fi
        


   # nodes:
   # node state: 2 = up, 4 = down, 32 = joining, 64 = leaving
   # resGroupNodeState: 2 = up, 4 = down
   if [ ! -z "$INDMZ" ]
   then
      ssh $HASNMPHOST "snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $DEFS clusternumnodes nodetable resGroupNodeState resGroupTable" > $BASE/snmp/$CLUSTER.nodeinfo 2>/dev/null
   else
      snmpinfo -m dump -t 1 -v -c $SNMPCOMMUNITY -h $HASNMPHOST -o $HASNMPHAMIB clusternumnodes nodetable resGroupNodeState resGroupTable > $BASE/snmp/$CLUSTER.nodeinfo 2>/dev/null
   fi

   echo Number of nodes: `grep -i clusternumnodes $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   state1=`grep -i ^nodestate.1 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   case $state1 in 
      2) nodestate1="UP" ;;
      4) nodestate1="DOWN" ;;
      32) nodestate1="JOINING" ;;
      64) nodestate1="LEAVING" ;;
   esac
   echo Node 1: `grep -i nodename.1 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"` $nodestate1
   state2=`grep -i ^nodestate.2 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   case $state2 in 
      2) nodestate2="UP" ;;
      4) nodestate2="DOWN" ;;
      32) nodestate2="JOINING" ;;
      64) nodestate2="LEAVING" ;;
   esac
   echo Node 2: `grep -i nodename.2 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"` $nodestate2
   state3=`grep -i ^nodestate.3 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   case $state3 in 
      2) nodestate3="UP" ;;
      4) nodestate3="DOWN" ;;
      32) nodestate3="JOINING" ;;
      64) nodestate3="LEAVING" ;;
   esac
   echo Node 3: `grep -i nodename.3 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"` $nodestate3
   state4=`grep -i ^nodestate.4 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   case $state4 in
      2) nodestate4="UP" ;;
      4) nodestate4="DOWN" ;;
      32) nodestate4="JOINING" ;;
      64) nodestate4="LEAVING" ;;
   esac
   echo Node 4: `grep -i nodename.4 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"` $nodestate4
   state5=`grep -i ^nodestate.5 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}'`
   case $state5 in
      2) nodestate5="UP" ;;
      4) nodestate5="DOWN" ;;
      32) nodestate5="JOINING" ;;
      64) nodestate5="LEAVING" ;;
   esac
   echo Node 5: `grep -i nodename.5 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"` $nodestate5

   resgroupnodestate1=`grep -i resGroupNodeState.1 $BASE/snmp/$CLUSTER.nodeinfo | grep "= 2" | awk '{print $1}' | cut -f3 -d'.'`
   resgroupnodestate2=`grep -i resGroupNodeState.2 $BASE/snmp/$CLUSTER.nodeinfo | grep "= 2" | awk '{print $1}' | cut -f3 -d'.'`
   resgroupnodestate3=`grep -i resGroupNodeState.3 $BASE/snmp/$CLUSTER.nodeinfo | grep "= 2" | awk '{print $1}' | cut -f3 -d'.'`
   resgroupname1=`grep -i resGroupName.1 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"`
   resgroupname2=`grep -i resGroupName.2 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"`
   resgroupname3=`grep -i resGroupName.3 $BASE/snmp/$CLUSTER.nodeinfo | awk '{print $3}' | sed "s/\"//g"`

   echo ResGrp1: $resgroupname1 $resgroupnodestate1
   echo ResGrp2: $resgroupname2 $resgroupnodestate2
   echo ResGrp3: $resgroupname3 $resgroupnodestate3

   # Clean up the trash.
   rm -f $BASE/snmp/interfacefile
   #rm -f $BASE/snmp/*clusterinfo $BASE/snmp/*nodeinfo
   unset nodestate1
   unset nodestate2
   unset nodestate3
   unset nodestate4
   unset nodestate5
   unset HASNMPHOST
   unset state1
   unset state2
   unset state3
   unset state4
   unset state5

fi

}

#
# MAIN
#

cd $BASE

# Create our temporary directory
mkdir $BASE/snmp 2>/dev/null

for clusternumber in `cat $BASE/clhosts | grep "^cluster:"  | cut -f2 -d:` 
do
   unset deemz
   deemz=`grep "^cluster:$clusternumber:" $BASE/clhosts | cut -f4 -d:`
   retrievehainfo $clusternumber $deemz > $BASE/snmp/$clusternumber.out
   ls -als $BASE/snmp/$clusternumber.out
done


